/* $Header$ */
// Contains the interface that allows a method called via the Omnis JavaScript worker to send its response back to Omnis
// Copyright (C) OLS Holdings Ltd 2018

/*********************************** Changes History **********************************
Date			Edit					Bug					Description
21-Aug-24	jmg_unmarked	ST/PF/1455	Throwing an error from an sync method killed Node.
21-May-24	jmg_unmarked							Added omnisModuleDefaultExport.
21-May-24	jmg_unmarked	ST/EC/1877	sendResponse() failed for data other than JS Objects.
23-Mar-19	jmg_unmarked							Updated sendResponse to base its response on the data that is passed in.
21-Nov-18	rmm_jsw										OW3 component - JavaScript Worker.
**************************************************************************************/
const http2 = require('node:http2');
const {errorCodes, newErrorWithCode} = require("./errors");

const omnis_calls = {

	/**
	 * Send any data as the response, attempting to automatically detect content type and convert data as necessary.
	 *
	 * @param data				The data to send back to Omnis.
	 * @param response		The 'response' object which was passed in to a module's 'call()' function.
	 */
	sendResponse: function(data, response) {

		if (!response) return false;

		let headers = {
			':status': 200,
		};

		let responseData = "";
		
		if (data === null || typeof data === "undefined") { //  Send empty data for null/undefined
			headers["Content-Type"] = "text/plain;charset=utf-8";
			responseData = "";
		}
		else if (data instanceof Buffer) { // Binary data
			headers["Content-Type"] = "application/octet-stream";
			responseData = data;
		}
		else if (data instanceof Object) { // Object or Array
			headers["Content-Type"] = "application/json;charset=utf-8";
			responseData = JSON.stringify(data);
		}
		else { // primitive type
			headers["Content-Type"] = "text/plain;charset=utf-8";
			responseData = data + ""; // Coerce into a string
		}

		response.respond(headers);
		response.end(responseData);

		return true;
	},

	/**
	 * Send a buffer with any specified content type as the response.
	 * Use to explicitly specify the content type and data to send.
	 *
	 * @param data					The data to send.
	 * @param contentType		The MIME type of the data to send. E.g. "image/png"
	 * @param {http2.Http2Stream} response			The 'response' object which was passed in to a module's 'call()' function.
	 */
	sendResponseBuffer: function(data, contentType, response) {

		if (!response) return false;

		response.respond({
			'content-type': contentType,
			':status': 200,
		});

		response.end(data);

		return true;
	},

	/**
	 * Send an error status response.
	 *
	 * @param {http2.Http2Stream} response		The 'response' object which was passed in to a module's 'call()' function.
	 * @param statusCode	An HTTP status code.
	 * @param errorText		A description of the error.
	 */
	sendError: function (response, statusCode, errorText) {
		
		if (!response) return false;

		response.respond({
			'content-type': 'text/plain; charset=utf-8',
			':status': statusCode,
		});

		response.end(errorText);

		return true;
	},

	/**
	 * Returns an object suitable for returning as the default export of an Omnis Module.
	 * 
	 * Supports async functions. Handles sending response to Omnis.
	 * Methods should just return the result object (or whatever) to send back to Omnis.
	 * 
	 * @param {Object} methodMap 	An object containing functions to expose as methods for your module. Key name is the name of the method.
	 */
	omnisModuleDefaultExport: function(methodMap) {
		return {
			call: function(method, param, response) { // The only requirement of an Omnis module is that it implement this function.
		
				if (!methodMap[method])
					throw newErrorWithCode(errorCodes.METHOD_NOT_FOUND, "");
		
				const result = methodMap[method](param, response);
				if (result instanceof Promise) {
					result.then((promiseResult) => {
						if (!response._writableState?.ended) // If response hasn't already been manually sent
							omnis_calls.sendResponse(promiseResult, response);
					})
					.catch((reason) => {
						if (!response._writableState?.ended)
							omnis_calls.sendError(response, 400, reason?.message);
					});
				}
				else {
					if (!response._writableState?.ended) // If response hasn't already been manually sent
						omnis_calls.sendResponse(result, response);
				}
		
				return true;
			}
		}
	}
};

module.exports = omnis_calls;
